unit GL520SM;

interface

uses
  windows, sysutils, MyUtils, SMB, Forms;

const
  ASUS_ID = $12C3;
  ASUS_CHIP_ID_99127F = $30;

  // GL520SM chip identifier
  WBOND_ID = $5CA3;
  WBOND_CHIP_ID_83781D = $10;
  WBOND_CHIP_ID_83782D = $30;
  WBOND_CHIP_ID_83783S = $40;
  WBOND_CHIP_ID_83627D = $20;
  WBOND_REG_CHIPID = $49;
  WBOND_REG_WCHIPID = $58;
  WBOND_REG_CHIPMAN = $4F;

  // GL520SM bank select & hi-lo ID select
  WBOND_HBACS = $80;
  WBOND_REG_BANK = $4E;
  WBOND_BANK0 = 0;
  WBOND_BANK1 = 1;
  WBOND_BANK2 = 2;
  WBOND_BANK3 = 3;
  WBOND_BANK4 = 4;
  WBOND_BANK5 = 5;
  WBOND_BANK6 = 6;
  WBOND_BANK7 = 7;

  // GL520SM ram value
  WBOND_REG_VID_FANDIV = $47;
  WBOND_REG_FAN1 = $28;
  WBOND_REG_FAN2 = $29;
  WBOND_REG_FAN3 = $2A;
  WBOND_REG_TEMP = $27;
  WBOND_REG_TEMP_OVER = $39;
  WBOND_REG_TEMP_HYST = $3A;
  WBOND_REG_TEMP2 = $0150;
  WBOND_REG_TEMP3 = $0250;
  WBOND_REG_TEMP2_HYST = $153;
  WBOND_REG_TEMP3_HYST = $253;
  WBOND_REG_TEMP2_CONFIG = $152;
  WBOND_REG_TEMP3_CONFIG = $252;
  WBOND_REG_TEMP2_OVER = $155;
  WBOND_REG_TEMP3_OVER = $255;
  WBOND_REG_CONFIG = $40;
  WBOND_REG_ALARM1 = $41;
  WBOND_REG_ALARM2 = $42;
  WBOND_REG_ALARM3 = $450;
  WBOND_REG_BEEP_CONFIG = $4D;
  WBOND_REG_BEEP_INTS1 = $56;
  WBOND_REG_BEEP_INTS2 = $57;
  WBOND_REG_BEEP_INTS3 = $453;
  WBOND_REG_PIN = $4B;
  WBOND_REG_5VSB = $50;
  WBOND_REG_VBAT = $51;
  WBOND_REG_PWM1 = $5B;
  WBOND_REG_PWM2 = $5A;
  WBOND_REG_PWM3 = $5E;
  WBOND_REG_PWM4 = $5F;
  WBOND_REG_PWMCLK12 = $5C;
  WBOND_REG_PWMCLK34 = $45C;
  WBOND_REG_I2C_ADDR = $48;
  WBOND_REG_I2C_SUBADDR = $4A;
  WBOND_REG_SCFG1 = $5D;
  WBOND_REG_RT_IDX = $50;
  WBOND_REG_RT_VAL = $51;

type
  Chip_Info = record
    Vendor_ID: word;
    Vendor_Name: string;
    Device_ID: byte;
    Device_Name: string;
    Rev: byte;
  end;
type
  Packet_Data = record
    IN0: string;
    IN1: string;
    IN2: string;
    IN3: string;
    IN4: string;
    IN5: string;
    IN6: string;
    IN7: string;
    IN8: string;
    TEMP1value: string;
    TEMP2value: string;
    TEMP3value: string;
    TEMP4value: string;
    TEMP5value: string;
    FAN1rpm: string;
    FAN2rpm: string;
    FAN3rpm: string;
    IN11: string;
    IN12: string;
    IN13: string;
    IN14: string;
    IN15: string;
    IN16: string;
    IN17: string;
    IN18: string;
    IN19: string;
  end;

  //   Chip_Structure:Chip_Info;

function GL520SM_Info(BaseAddr: dword; Chip_Addr: byte): Chip_Info;
function GL520SM_Set_Bank(BaseAddr: dword; Chip_Addr: byte; Bank: byte): boolean;
function GL520SM_Chip(BaseAddr: dword; Chip_Addr: byte): Packet_Data;

var
  ASUS_FLAG: boolean;
  WBOND_FLAG: smallint;

implementation

function GL520SM_Chip(BaseAddr: dword; Chip_Addr: byte): Packet_Data;
{var
  value_reg: Packet_Data;
  Dump: word;
  decimal, dmp: real;
  datat: integer;
  div1, div2: word;
}begin
{  Application.ProcessMessages;
  GL520SM_Set_Bank(BaseAddr, Chip_Addr, WBOND_BANK0);
  value_reg.IN0 := FormatFloat('#0.00', (smbGetReg(BaseAddr, $20, Chip_Addr) * 0.016));
  value_reg.IN1 := FormatFloat('#0.00', (smbGetReg(BaseAddr, $21, Chip_Addr) * 0.016));
  value_reg.IN2 := FormatFloat('#0.00', (smbGetReg(BaseAddr, $22, Chip_Addr) * 0.016));
  value_reg.IN3 := FormatFloat('#0.00', (smbGetReg(BaseAddr, $23, Chip_Addr) * 0.016 * 1.68));
  if ASUS_FLAG then
    value_reg.IN4 := FormatFloat('#0.00', (smbGetReg(BaseAddr, $24, Chip_Addr) * 0.016 * 4.028))
  else
    value_reg.IN4 := FormatFloat('#0.00', (smbGetReg(BaseAddr, $24, Chip_Addr) * 0.016 * 3.80));
  datat := smbGetReg(BaseAddr, $25, Chip_Addr);
  if ASUS_FLAG then
    value_reg.IN5 := FormatFloat('#0.00',-datat * 0.016 * 1.73)
    else if (WBOND_FLAG = 2) then
      value_reg.IN5 := FormatFloat('#0.00', (datat * 0.016*-3.6 * 0.8112) / 0.34)
    else
      value_reg.IN5 := FormatFloat('#0.00',-datat * 0.016 * 3.477);
  datat := smbGetReg(BaseAddr, $26, Chip_Addr);
  if ASUS_FLAG then
    value_reg.IN6 := FormatFloat('#0.00',-datat * 0.016 * 3.605)
    else if (WBOND_FLAG = 2) then
      value_reg.IN6 := FormatFloat('#0.00', (datat * 0.016*-3.6 * 0.6118) / 0.16)
    else
      value_reg.IN6 := FormatFloat('#0.00',-datat * 0.016 * 1.505);
  Dump := smbGetReg(BaseAddr, WBOND_REG_VID_FANDIV, Chip_Addr);
  div1 := 1 shl ((Dump and $30) shr 4);
  div2 := 1 shl ((Dump and $C0) shr 6);
  Dump := smbGetReg(BaseAddr, WBOND_REG_FAN1, Chip_Addr) and $FF;
  if (Dump <> $FF) and (Dump <> $0) then
    value_reg.FAN1rpm := floatToStr(trunc(1350000 / (Dump * div1)))
  else
    value_reg.FAN1rpm := 'n.p.';
  Dump := smbGetReg(BaseAddr, WBOND_REG_FAN2, Chip_Addr) and $FF;
  if (Dump <> $FF) and (Dump <> $0) then
    value_reg.FAN2rpm := floatToStr(trunc(1350000 / (Dump * div2)))
  else
    value_reg.FAN2rpm := 'n.p.';
  Dump := smbGetReg(BaseAddr, WBOND_REG_FAN3, Chip_Addr);
  if (Dump <> $FF) and (Dump <> $0) then
    value_reg.FAN3rpm := IntToStr(trunc(1350000 / (Dump * 2)))
  else
    value_reg.FAN3rpm := 'n.p.';
  Dump := smbGetReg(BaseAddr, WBOND_REG_TEMP, Chip_Addr);
  if Dump < 100 then
    value_reg.TEMP1value := FormatFloat('#00.0', Dump + 8) + 'C'
  else
    value_reg.TEMP1value := '0.0 C';
  GL520SM_Set_Bank(BaseAddr, Chip_Addr, WBOND_BANK1);
  decimal := (smbGetReg(BaseAddr, $51, Chip_Addr) shr 7 and 1) * 0.5;
  //  GL520SM_Set_Bank(BaseAddr, Chip_Addr, WBOND_BANK1);
  Dump := smbGetReg(BaseAddr, $50, Chip_Addr);
  if Dump < 100 then
    value_reg.TEMP2value := FormatFloat('#0.0', Dump + decimal) + 'C'
  else
    value_reg.TEMP2value := '0.0C';
  GL520SM_Set_Bank(BaseAddr, Chip_Addr, WBOND_BANK2);
  decimal := (smbGetReg(BaseAddr, $51, Chip_Addr) shr 7 and 1) * 0.5;
  //  GL520SM_Set_Bank(BaseAddr, Chip_Addr, WBOND_BANK1);
  Dump := smbGetReg(BaseAddr, $50, Chip_Addr);
  if Dump < 100 then
    value_reg.TEMP3value := FormatFloat('#0.0', Dump + decimal) + 'C'
  else
    value_reg.TEMP3value := FormatFloat('#0.0', 0) + 'C';
  GL520SM_Set_Bank(BaseAddr, Chip_Addr, WBOND_BANK0);
  Dump := smbGetReg(BaseAddr, $0, $48);
  if Dump < 100 then
    value_reg.TEMP4value := FormatFloat('#0.0', Dump + decimal) + 'C'
  else
    value_reg.TEMP4value := FormatFloat('#0.0', 0) + 'C';
  Dump := smbGetReg(BaseAddr, $0, $49);
  if Dump < 100 then
    value_reg.TEMP5value := FormatFloat('#0.0', Dump + decimal) + 'C'
  else
    value_reg.TEMP5value := FormatFloat('#0.0', 0) + 'C';
  GL520SM_Set_Bank(BaseAddr, Chip_Addr, WBOND_BANK5);
  if ASUS_FLAG then
    value_reg.IN7 := 'n.p.'
  else begin
    dmp := smbGetReg(BaseAddr, WBOND_REG_VBAT, Chip_Addr) * 0.016;
    value_reg.IN7 := FormatFloat('#0.00', dmp);
  end;
  dmp := smbGetReg(BaseAddr, WBOND_REG_5VSB, Chip_Addr) * 0.016 * 1.68;
  if ASUS_FLAG then
    value_reg.IN8 := 'n.p.'
  else
    value_reg.IN8 := FormatFloat('#0.00', dmp);
  GL520SM_Set_Bank(BaseAddr, Chip_Addr, WBOND_BANK0);
  Result := value_reg;
}end;

//------------------------------------------------------------------------------
//  GL520SM_SET_BANK  input:  BaseAddr     : dWord
//                            Chip_Addr    : byte
//                            Bank         : byte
//                   output:  boolean (TRUE if the bank has been selected)
//
//  Utilizza le sub GET_REG e SET_REG nel modulo SMB.PAS
//
//  Seleziona il BANK REGISTER su cui lavorare, ritorna VERO se l'operazione
//  ha avuto successo, FALSO altrimenti.
//
//------------------------------------------------------------------------------
function GL520SM_Set_Bank(BaseAddr: dword; Chip_Addr: byte; Bank: byte): boolean;
{var
  Data: byte;
}begin
{  Data := smbGetReg(BaseAddr, WBOND_REG_BANK, Chip_Addr);
  Bank := (Bank and 7) or (Data and WBOND_HBACS);
  smbSetReg(BaseAddr, WBOND_REG_BANK, Chip_Addr, Bank);
  Data := smbGetReg(BaseAddr, WBOND_REG_BANK, Chip_Addr);
  if Data = Bank then
    Result := True
  else              }
    Result := false;
end;

//------------------------------------------------------------------------------
//  GL520SM_INFO      input:  BaseAddr     : dWord
//                            Chip_Addr    : byte
//                   output:  CHIP_INFO (Typed)
//
//  Utilizza le sub GET_REG e SET_REG nel modulo SMB.PAS
//
//  Ritorna una struttura contenente le informazioni sul chip SMB del tipo:
//    Vendor_ID    :word;
//    Vendor_Name  :string;
//    Device_ID    :byte;
//    Device_Name  :string;
//    Rev          :byte;
//
//------------------------------------------------------------------------------
function GL520SM_Info(BaseAddr: dword; Chip_Addr: byte): Chip_Info;
var
  Info: Chip_Info;
  Data, temp: cardinal;
  Dump: string;
begin
  Data := smbGetReg(BaseAddr, $00, Chip_Addr);
  Info.Vendor_ID := data;
  temp:=Data;
  Data := smbGetReg(BaseAddr, $01, Chip_Addr);
  Info.Device_ID := Data and $FE;
  Dump := Format('%x', [temp]);
  case temp of
    $20:
      begin
        Info.Device_Name := 'GL520SM';
        Info.Vendor_Name := 'Genesys';
      end;
  end;
  Result := Info;
end;

end.
